	function [poles,zero_angles, scale] = ClassicIIR_LP_poleszeros(N, A_p,...
	omega_p,omega_s, iirtype)
	
	% Given 
	% 1) the low-pass filter type:
	%    iirtype=1 for Butterworth,
	%    iirtype=2 for Chebyshev or Chebyshev Type I,
	%    iirtype=3 for inverse Chebyshev or Chebyshev Type II, and
	%    iirtype=4 for elliptic or Cauer,
	% 2) N, the filter order, omega_p, the passband edge as a fraction of pi or
	%    half the sampling rate, and A_p, the passband variation in dB, as well
	%    as
	% 3) omega_s, the stopband edge as a fraction of pi or half the 
	%    sampling rate for iirtype=3 (inverse Chebyshev) and iirtype=4 (Cauer),
	% [poles,zero_angles, scale]=ClassicIIR_LP_polezero(N, A_p, omega_p,...
	%    omega_s, iirtype)
	% returns the filter poles, included in the vector poles, the angles of 
	% the filter zeros, included in the vector zero_angles, and the scaling 
	% constant scale.  
	% This file has been constructed based on the article
	% S. Darlington, "Simple algorithms for elliptic filters and 
	% generalizations thereof", IEEE Trans. Circuits and Systems, CAS-25, 
	% pp. 975 - 980, Dec. 1978.
	% as well as the lecture notes
	% http://www.cs.tut.fi/~ts/part4.pdf.
	% Tapio, 11.9.2018
	% A new reference is coming up soon:
	% T. Saramki, "Simple Formulae for Designing Classical Recursive Digital
	% Filters".
	%------------------------------------------------------------
	
	% 	Toolbox for DIGITAL FILTERS USING MATLAB 
	
	%	Author: 			Tapio Saramaki 2018-09-11
	%	Modified by:	
	%	Copyright:			by authors - not released for commercial use
	%	Version:			1	 
	%	Known bugs:			None
	%	Report bugs to:		tapio.saramaki@tut.fi
	
	
	%**************************************************************************
	%                        Butterworth Filter
	%**************************************************************************
	if iirtype==1
	    k=1:N;
	    pol=tan(pi*omega_p/2)*(1/(10^(A_p/10)-1))^(1/(2*N))*...
		exp(j*pi*(1/2+(2*k-1)/(2*N))); 
	    poles=(1+pol)./(1-pol);
	    zero_angles(k)=pi;
	    scale=real(prod(1-poles))/real(prod(1-exp(j*zero_angles)));
	end
	%**************************************************************************
	%                        Chebyshev Filter
	%**************************************************************************
	if iirtype==2
	    k=1:N;
	    polhat=(sqrt(1/(10^(A_p/10)-1))+sqrt(1/(10^(A_p/10)-1)+1))^(1/N)*...
		exp(j*pi*(1/2+(2*k-1)/(2*N)));
	    pol=tan(pi*omega_p/2)*(polhat-1./polhat)/2;
	    poles=(1+pol)./(1-pol);
	    zero_angles(k)=pi;
	    if rem(N,2)==0 
	        scale=sqrt(10^(-A_p/10));
		else
	        scale=1;
		end
	    scale=scale*real(prod(1-poles))/real(prod(1-exp(j*zero_angles)));
	end
	%**************************************************************************
	%                        Inverse Chebyshev Filter
	%**************************************************************************
	if iirtype==3
	    %----------------------------------------------------------------------
	    % It turned out that the poles can be conveniently determined with the 
	    % aid of a Chebyshev filter in the following four steps:
	    %
	    % Step 1: Determine the stopband attenuation A_s.
	    %
	    a0=sqrt(tan(pi*omega_s/2)/tan(pi*omega_p/2));  
	    a1=a0^2+sqrt(a0^4-1);
	    A_s=10*log10(1+(10^(A_p/10)-1)*((a1^N+1/a1^N)/2)^2);
	    %
	    % Step 2: Determine the passband variation for the power-complementary 
	    %         filter.
	    %
	    Acomp_p=-10*log10(1-10^(-A_s/10));
	    %
	    % Step 3: Determine the poles of the Chebyshev filter whose passband 
	    %         variation is Acomp_p and passband edge is 
	    %         omeganew_p=1-omega_s.
	    %
	    omeganew_p=1-omega_s;
	    k=1:N;
	    polhat=(sqrt(1/(10^(Acomp_p/10)-1))+sqrt(1/(10^(Acomp_p/10)-1)+1))^(1/N)*...
		exp(j*pi*(1/2+(2*k-1)/(2*N)));
	    pol=tan(pi*omeganew_p/2)*(polhat-1./polhat)/2;
	    poles=(1+pol)./(1-pol);
	    %
	    % Step 4: Perform the lowpass-to-highpass transformation 
	    % z^{-1}--> -z^{-1} by changing the signs of the real parts.
	    %
	    poles=-real(poles)+j*imag(poles);
	    %
	    % The zeros and the scaling constant are determined in a convenient
	    % manner: Perhaps some references are needed.
	    %
	    kk=1:floor(N/2);
	    zer=tan(pi*omega_s/2)./cos(pi*(2*kk-1)/(2*N));
	    zero_angles=2*atan(zer);
	    if rem(N,2)==1
	        zero_angles=[zero_angles pi -flip(zero_angles)];
		else
	        zero_angles=[zero_angles -flip(zero_angles)];
		end
	    scale=real(prod(1-poles))/real(prod(1-exp(j*zero_angles)));
	end
	%**************************************************************************
	%                        Cauer Filter
	%**************************************************************************
	if iirtype==4
	    %**********
	    % Poles
	    %***********
	    a0=sqrt(tan(pi*omega_s/2)/tan(pi*omega_p/2));  
	    a1=a0^2+sqrt(a0^4-1);
	    a2=a1^2+sqrt(a1^4-1);
	    a3=a2^2+sqrt(a2^4-1);
	    a4=a3^2+sqrt(a3^4-1);
	    J4=2^(N-1)*a4^N;
	    J3=sqrt((J4+1/J4)/2);
	    J2=sqrt((J3+1/J3)/2);
	    J1=sqrt((J2+1/J2)/2);
	    S10=sqrt(1/(10^(A_p/10)-1))+sqrt(1/(10^(A_p/10)-1)+1);
	    S20=J1*S10+sqrt((J1*S10)^2+1);
	    S30=J2*S20+sqrt((J2*S20)^2+1);
	    S40=J3*S30+sqrt((J3*S30)^2+1);
	    k=1:N;
	    pol5=(J4/S40+sqrt((J4/S40)^2+1))^(1/N)*...
		exp(j*pi*(1/2-(2*k-1)/(2*N)));
	    pol4=(pol5-1./pol5)/(2*a4);
	    pol3=(pol4-1./pol4)/(2*a3);
	    pol2=(pol3-1./pol3)/(2*a2);
	    pol1=(pol2-1./pol2)/(2*a1);
	    pol=tan(pi*omega_p/2)*(pol1-1./pol1)/2;
	    poles=(1+pol)./(1-pol);
	    %************
	    % Zeros
	    %************
	    kk=1:floor(N/2);
	    zer4=a4./(cos(pi*(2*kk-1)/(2*N)));
	    zer3=(zer4+1./zer4)/(2*a3);
	    zer2=(zer3+1./zer3)/(2*a2);
	    zer1=(zer2+1./zer2)/(2*a1);
	    zer0=tan(pi*omega_p/2)*(zer1+1./zer1)/2;
	    zero_angles=2*atan(zer0);    
	    if rem(N,2)==1
	        zero_angles=[zero_angles pi -flip(zero_angles)];
		else
	        zero_angles=[zero_angles -flip(zero_angles)];
		end
	    %*******************
	    % Scaling constant
	    %*******************
	    if rem(N,2)==0 
	        scale=sqrt(10^(-A_p/10));
		else
	        scale=1;
		end
	    scale=scale*real(prod(1-poles))/real(prod(1-exp(j*zero_angles)));
	end
